<?php

namespace App\Services;

use App\Models\FormResponse;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class WhatsAppService
{
    /**
     * Send form submission notification via WhatsApp
     */
    public function sendFormSubmission(string $toNumber, FormResponse $formResponse): bool
    {
        $message = $this->formatFormSubmissionMessage($formResponse);
        
        return $this->sendMessage($toNumber, $message);
    }

    /**
     * Format form submission data into WhatsApp message
     */
    private function formatFormSubmissionMessage(FormResponse $formResponse): string
    {
        $data = $formResponse->data;
        
        $message = "🎯 New Application Received\n\n";
        $message .= "Reference ID: {$formResponse->reference_id}\n\n";
        
        if (isset($data['name'])) {
            $message .= "Full Name: {$data['name']}\n";
        }
        
        if (isset($data['phone'])) {
            $message .= "Phone Number: {$data['phone']}\n";
        }
        
        if (isset($data['nationality'])) {
            $message .= "Nationality: {$data['nationality']}\n";
        }
        
        if (isset($data['gender'])) {
            $message .= "Gender: {$data['gender']}\n";
        }
        
        if (isset($data['position'])) {
            $message .= "Position Applied For: {$data['position']}\n";
        }
        
        if (isset($data['experience'])) {
            $message .= "Years of Experience: {$data['experience']}\n";
        }
        
        if (isset($data['companies'])) {
            $message .= "Previous Companies: {$data['companies']}\n";
        }
        
        if (isset($data['salary'])) {
            $message .= "Expected Salary (AED): {$data['salary']}\n";
        }
        
        if (isset($data['in_dubai'])) {
            $message .= "Currently in Dubai?: {$data['in_dubai']}\n";
        }
        
        if (isset($data['english_level'])) {
            $message .= "English Level: {$data['english_level']}\n";
        }
        
        if (isset($data['notice_period'])) {
            $message .= "Notice Period: {$data['notice_period']}\n";
        }
        
        $message .= "\n---\n";
        $message .= "Submitted on: " . $formResponse->created_at->format('d M Y, h:i A');
        
        return $message;
    }

    /**
     * Send WhatsApp message using configured API
     * 
     * This is a placeholder implementation. You'll need to integrate with your
     * WhatsApp Business API provider (e.g., Twilio, MessageBird, WhatsApp Cloud API)
     */
    private function sendMessage(string $toNumber, string $message): bool
    {
        $apiUrl = config('services.whatsapp.api_url');
        $apiKey = config('services.whatsapp.api_key');
        $fromNumber = config('services.whatsapp.from_number');
        
        if (!$apiUrl || !$apiKey) {
            Log::warning('WhatsApp API not configured. Message would have been sent to: ' . $toNumber);
            Log::info('WhatsApp Message: ' . $message);
            return true; // Return true to not fail form submission
        }

        try {
            // Example using WhatsApp Cloud API format
            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $apiKey,
                'Content-Type' => 'application/json',
            ])->post($apiUrl, [
                'messaging_product' => 'whatsapp',
                'to' => $toNumber,
                'type' => 'text',
                'text' => [
                    'body' => $message,
                ],
            ]);

            if ($response->successful()) {
                Log::info('WhatsApp message sent successfully to: ' . $toNumber);
                return true;
            }

            Log::error('WhatsApp API error: ' . $response->body());
            return false;
        } catch (\Exception $e) {
            Log::error('WhatsApp send exception: ' . $e->getMessage());
            return false;
        }
    }
}
