<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;

class FormResponse extends Model
{
    use HasFactory;

    protected $fillable = [
        'reference_id',
        'data',
        'whatsapp_number',
        'viewed',
        'viewed_at',
        'viewed_by',
    ];

    protected $casts = [
        'data' => 'array',
        'viewed' => 'boolean',
        'viewed_at' => 'datetime',
    ];

    public function viewedBy(): BelongsTo
    {
        return $this->belongsTo(User::class, 'viewed_by');
    }

    protected static function boot()
    {
        parent::boot();

        static::creating(function ($formResponse) {
            if (empty($formResponse->reference_id)) {
                $formResponse->reference_id = self::generateReferenceId();
            }
        });
    }

    public static function generateReferenceId(): string
    {
        // Get the latest form response
        $latestResponse = self::orderBy('id', 'desc')->first();
        
        // Extract the number from the latest reference_id or start from 1
        if ($latestResponse && $latestResponse->reference_id) {
            // Handle both formats: "REF-000001" or numeric "12345"
            if (preg_match('/REF-(\d+)/', $latestResponse->reference_id, $matches)) {
                $nextNumber = (int) $matches[1] + 1;
            } else {
                $nextNumber = (int) $latestResponse->reference_id + 1;
            }
        } else {
            $nextNumber = 1;
        }
        
        // Format as REF-XXXXXX (6 digits with leading zeros)
        return 'REF-' . str_pad($nextNumber, 6, '0', STR_PAD_LEFT);
    }
}
