<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Http\Requests\FormSubmitRequest;
use App\Models\FormConfig;
use App\Models\FormResponse;
use App\Services\WhatsAppService;
use Illuminate\Http\JsonResponse;

class FormController extends Controller
{
    public function __construct(
        private WhatsAppService $whatsAppService
    ) {}

    /**
     * Submit recruitment form data
     */
    public function submit(FormSubmitRequest $request): JsonResponse
    {
        try {
            $validated = $request->validated();
            
            // Get current form config for WhatsApp number
            $config = FormConfig::latest()->first();
            
            if (!$config) {
                return response()->json([
                    'success' => false,
                    'error' => 'Form configuration not found',
                    'code' => 'CONFIG_NOT_FOUND',
                ], 404);
            }

            // Create form response
            $formResponse = FormResponse::create([
                'data' => $validated['data'],
                'whatsapp_number' => $validated['data']['phone'] ?? '',
            ]);

            // Send WhatsApp message
            try {
                $this->whatsAppService->sendFormSubmission(
                    $config->whatsapp_number,
                    $formResponse
                );
            } catch (\Exception $e) {
                \Log::error('WhatsApp send failed: ' . $e->getMessage());
                // Don't fail the submission if WhatsApp fails
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'id' => (string) $formResponse->id,
                    'referenceId' => $formResponse->reference_id,
                    'timestamp' => $formResponse->created_at->toIso8601String(),
                    'whatsappNumber' => $config->whatsapp_number,
                    'message' => 'Form submitted successfully',
                ],
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to submit form',
                'code' => 'SUBMISSION_ERROR',
            ], 500);
        }
    }
}
