<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\BulkMarkViewedRequest;
use App\Http\Requests\MarkViewedRequest;
use App\Models\FormResponse;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;

class ResponseController extends Controller
{
    /**
     * Get all form submissions with optional filtering
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $limit = $request->input('limit', 100);
            $offset = $request->input('offset', 0);
            $viewed = $request->input('viewed', 'no'); // Default to 'no' (unviewed)
            $sortBy = $request->input('sortBy', 'created_at');
            $sortOrder = $request->input('sortOrder', 'desc');

            $query = FormResponse::query();

            // Filter by viewed status
            if (strtolower($viewed) !== 'all') {
                // Handle 'yes', 'no', 'true', 'false', '1', '0'
                $viewedValue = in_array(strtolower($viewed), ['yes', 'true', '1']);
                $query->where('viewed', $viewedValue);
            }
            // If viewed is 'all', don't filter by viewed status

            // Filter by form field data (name, phone, email, nationality, etc.)
            $exactMatchFields = ['gender', 'in_dubai']; // Fields that require exact match
            $likeMatchFields = ['name', 'phone', 'email', 'nationality', 'position', 
                               'experience', 'companies', 'salary', 'english_level', 'notice_period'];
            
            // Handle exact match fields
            foreach ($exactMatchFields as $field) {
                if ($request->has($field)) {
                    $value = $request->input($field);
                    $query->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(data, '$.{$field}')) = ?", [$value]);
                }
            }
            
            // Handle LIKE match fields
            foreach ($likeMatchFields as $field) {
                if ($request->has($field)) {
                    $value = $request->input($field);
                    $query->whereRaw("JSON_UNQUOTE(JSON_EXTRACT(data, '$.{$field}')) LIKE ?", ["%{$value}%"]);
                }
            }

            // Get total count before pagination
            $total = $query->count();

            // Apply sorting
            $query->orderBy($sortBy, $sortOrder);

            // Apply pagination
            $responses = $query->skip($offset)->take($limit)->get();

            // Format responses
            $formattedResponses = $responses->map(function ($response) {
                return [
                    'id' => (string) $response->id,
                    'referenceId' => $response->reference_id,
                    'timestamp' => $response->created_at->toIso8601String(),
                    'viewed' => $response->viewed,
                    'viewedAt' => $response->viewed_at?->toIso8601String(),
                    'data' => $response->data,
                    'whatsappNumber' => $response->whatsapp_number,
                ];
            });

            // Calculate stats
            $totalResponses = FormResponse::count();
            $viewedResponses = FormResponse::where('viewed', true)->count();
            $unviewedResponses = $totalResponses - $viewedResponses;

            return response()->json([
                'success' => true,
                'data' => [
                    'responses' => $formattedResponses,
                    'pagination' => [
                        'total' => $total,
                        'limit' => $limit,
                        'offset' => $offset,
                        'hasMore' => ($offset + $limit) < $total,
                    ],
                    'stats' => [
                        'totalResponses' => $totalResponses,
                        'viewedResponses' => $viewedResponses,
                        'unviewedResponses' => $unviewedResponses,
                    ],
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to fetch responses',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Get single response by ID
     */
    public function show(string $id): JsonResponse
    {
        try {
            $response = FormResponse::find($id);

            if (!$response) {
                return response()->json([
                    'success' => false,
                    'error' => 'Response not found',
                    'code' => 'NOT_FOUND',
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'id' => (string) $response->id,
                    'referenceId' => $response->reference_id,
                    'timestamp' => $response->created_at->toIso8601String(),
                    'viewed' => $response->viewed,
                    'viewedAt' => $response->viewed_at?->toIso8601String(),
                    'viewedBy' => $response->viewed_by ? (string) $response->viewed_by : null,
                    'data' => $response->data,
                    'whatsappNumber' => $response->whatsapp_number,
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to fetch response',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Mark single response as viewed
     */
    public function markViewed(string $id, MarkViewedRequest $request): JsonResponse
    {
        try {
            $response = FormResponse::find($id);

            if (!$response) {
                return response()->json([
                    'success' => false,
                    'error' => 'Response not found',
                    'code' => 'NOT_FOUND',
                ], 404);
            }

            $validated = $request->validated();
            
            $response->update([
                'viewed' => $validated['viewed'],
                'viewed_at' => $validated['viewed'] ? now() : null,
                'viewed_by' => $validated['viewed'] ? auth()->id() : null,
            ]);

            return response()->json([
                'success' => true,
                'data' => [
                    'id' => (string) $response->id,
                    'viewed' => $response->viewed,
                    'viewedAt' => $response->viewed_at?->toIso8601String(),
                    'viewedBy' => $response->viewed_by ? (string) $response->viewed_by : null,
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to update response',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Bulk mark responses as viewed/unviewed
     */
    public function bulkMarkViewed(BulkMarkViewedRequest $request): JsonResponse
    {
        try {
            $validated = $request->validated();
            $ids = $validated['ids'];
            $viewed = $validated['viewed'];

            $updated = [];
            $failed = [];
            $errors = [];

            foreach ($ids as $id) {
                $response = FormResponse::find($id);

                if (!$response) {
                    $failed[] = $id;
                    $errors[$id] = 'Response not found';
                    continue;
                }

                try {
                    $response->update([
                        'viewed' => $viewed,
                        'viewed_at' => $viewed ? now() : null,
                        'viewed_by' => $viewed ? auth()->id() : null,
                    ]);
                    $updated[] = $id;
                } catch (\Exception $e) {
                    $failed[] = $id;
                    $errors[$id] = 'Failed to update';
                }
            }

            $result = [
                'updated' => count($updated),
                'failed' => count($failed),
                'ids' => $updated,
                'failedIds' => $failed,
                'updatedAt' => now()->toIso8601String(),
            ];

            if (!empty($errors)) {
                $result['errors'] = $errors;
            }

            return response()->json([
                'success' => true,
                'data' => $result,
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to bulk update responses',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }
}
