<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Models\Interviews;
use App\Models\FormResponse;
use App\Models\WhatsAppMessage;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;

class InterviewsController extends Controller
{
    /**
     * Get all interviews
     */
    public function index(Request $request): JsonResponse
    {
        try {
            $page = max(1, (int) $request->input('page', 1));
            $perPage = max(1, min(100, (int) $request->input('per_page', 15)));
            $status = $request->input('status');
            $candidateStatus = $request->input('candidate_status');
            $name = $request->input('name');
            $referenceId = $request->input('referenceId');
            $date = $request->input('date');

            $query = Interviews::query();

            // Filter by status
            if ($status !== null) {
                $query->where('status', $status);
            }

            // Filter by candidate_status
            if ($candidateStatus !== null) {
                $query->where('candidate_status', $candidateStatus);
            }

            // Filter by candidate name (partial match)
            if ($name !== null) {
                $query->where('candidate_name', 'LIKE', '%' . $name . '%');
            }

            // Filter by reference ID (exact match)
            if ($referenceId !== null) {
                $query->where('reference_id', $referenceId);
            }

            // Filter by date
            if ($date !== null) {
                // If specific date is provided, filter by that date
                $query->whereDate('date', $date);
            } else {
                // If no date specified, show today and future dates only
                $query->whereDate('date', '>=', now()->toDateString());
            }

            // Get total count before pagination
            $total = $query->count();

            // Apply sorting - order by date ascending, then time ascending
            $query->orderBy('date', 'asc')->orderBy('time', 'asc');

            // Calculate pagination
            $totalPages = ceil($total / $perPage);
            $offset = ($page - 1) * $perPage;

            // Apply pagination
            $interviews = $query->skip($offset)->take($perPage)->get();

            // Format interviews
            $formattedInterviews = $interviews->map(function ($interview) {
                return [
                    'id' => (string) $interview->id,
                    'response_id' => (string) $interview->response_id,
                    'referenceId' => $interview->reference_id,
                    'candidate_name' => $interview->candidate_name,
                    'name' => $interview->candidate_name,
                    'phone' => $interview->phone,
                    'date' => $interview->date->format('Y-m-d'),
                    'time' => $interview->time->format('H:i'),
                    'status' => $interview->status,
                    'candidate_status' => $interview->candidate_status,
                    'whatsapp_status' => $interview->whatsapp_status,
                    'whatsapp_sent_at' => $interview->whatsapp_sent_at?->format('Y-m-d H:i:s'),
                    'whatsapp_delivered_at' => $interview->whatsapp_delivered_at?->format('Y-m-d H:i:s'),
                    'whatsapp_read_at' => $interview->whatsapp_read_at?->format('Y-m-d H:i:s'),
                ];
            });

            return response()->json([
                'success' => true,
                'data' => [
                    'interviews' => $formattedInterviews,
                    'pagination' => [
                        'total' => $total,
                        'per_page' => $perPage,
                        'current_page' => $page,
                        'total_pages' => $totalPages,
                        'from' => $total > 0 ? $offset + 1 : 0,
                        'to' => min($offset + $perPage, $total),
                        'has_more' => $page < $totalPages,
                    ],
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to fetch interviews',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Create a new interview
     */
    public function store(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'response_id' => 'required|exists:form_responses,id',
                'reference_id' => 'required|exists:form_responses,reference_id',
                'candidate_name' => 'required|string|max:255',
                'phone' => 'required|string|max:50',
                'date' => 'required|date',
                'time' => 'required|date_format:H:i',
                'status' => 'nullable|string',
                'candidate_status' => 'nullable|string',
                'user_id' => 'nullable|exists:users,id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'error' => 'Validation failed',
                    'code' => 'VALIDATION_ERROR',
                    'details' => $validator->errors()->toArray(),
                ], 400);
            }

            $validated = $validator->validated();
            $validated['user_id'] = $validated['user_id'] ?? auth()->id();
            $validated['status'] = 'pending';

            $interview = Interviews::create($validated);

            return response()->json([
                'success' => true,
                'data' => [
                    'id' => (string) $interview->id,
                    'response_id' => (string) $interview->response_id,
                    'referenceId' => $interview->reference_id,
                    'candidate_name' => $interview->candidate_name,
                    'name' => $interview->candidate_name,
                    'phone' => $interview->phone,
                    'date' => $interview->date->format('Y-m-d'),
                    'time' => $interview->time->format('H:i'),
                    'status' => $interview->status,
                    'candidate_status' => $interview->candidate_status,
                ],
            ], 201);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to create interview',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Get a single interview
     */
    public function show(string $id): JsonResponse
    {
        try {
            $interview = Interviews::find($id);

            if (!$interview) {
                return response()->json([
                    'success' => false,
                    'error' => 'Interview not found',
                    'code' => 'NOT_FOUND',
                ], 404);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'id' => (string) $interview->id,
                    'response_id' => (string) $interview->response_id,
                    'referenceId' => $interview->reference_id,
                    'candidate_name' => $interview->candidate_name,
                    'name' => $interview->candidate_name,
                    'phone' => $interview->phone,
                    'date' => $interview->date->format('Y-m-d'),
                    'time' => $interview->time->format('H:i'),
                    'status' => $interview->status,
                    'candidate_status' => $interview->candidate_status,
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to fetch interview',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Update an interview
     */
    public function update(Request $request, string $id): JsonResponse
    {
        try {
            $interview = Interviews::find($id);

            if (!$interview) {
                return response()->json([
                    'success' => false,
                    'error' => 'Interview not found',
                    'code' => 'NOT_FOUND',
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'candidate_name' => 'nullable|string|max:255',
                'phone' => 'nullable|string|max:50',
                'date' => 'nullable|date',
                'time' => 'nullable|date_format:H:i',
                'status' => 'nullable|string',
                'candidate_status' => 'nullable|string',
                'user_id' => 'nullable|exists:users,id',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'error' => 'Validation failed',
                    'code' => 'VALIDATION_ERROR',
                    'details' => $validator->errors()->toArray(),
                ], 400);
            }

            $interview->update($validator->validated());

            return response()->json([
                'success' => true,
                'data' => [
                    'id' => (string) $interview->id,
                    'response_id' => (string) $interview->response_id,
                    'referenceId' => $interview->reference_id,
                    'candidate_name' => $interview->candidate_name,
                    'name' => $interview->candidate_name,
                    'phone' => $interview->phone,
                    'date' => $interview->date->format('Y-m-d'),
                    'time' => $interview->time->format('H:i'),
                    'status' => $interview->status,
                    'candidate_status' => $interview->candidate_status,
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to update interview',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Delete an interview
     */
    public function destroy(string $id): JsonResponse
    {
        try {
            $interview = Interviews::find($id);

            if (!$interview) {
                return response()->json([
                    'success' => false,
                    'error' => 'Interview not found',
                    'code' => 'NOT_FOUND',
                ], 404);
            }

            $interview->delete();

            return response()->json([
                'success' => true,
                'data' => [
                    'message' => 'Interview deleted successfully',
                    'id' => $id,
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to delete interview',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Update candidate status for an interview
     */
    public function updateCandidateStatus(Request $request, string $id): JsonResponse
    {
        try {
            $interview = Interviews::find($id);

            if (!$interview) {
                return response()->json([
                    'success' => false,
                    'error' => 'Interview not found',
                    'code' => 'NOT_FOUND',
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'candidate_status' => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'error' => 'Validation failed',
                    'code' => 'VALIDATION_ERROR',
                    'details' => $validator->errors()->toArray(),
                ], 400);
            }

            $interview->update([
                'candidate_status' => $request->input('candidate_status'),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Candidate status updated successfully',
                'interview' => [
                    'id' => (string) $interview->id,
                    'candidate_status' => $interview->candidate_status,
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to update candidate status',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Update interview status
     */
    public function updateStatus(Request $request, string $id): JsonResponse
    {
        try {
            $interview = Interviews::find($id);

            if (!$interview) {
                return response()->json([
                    'success' => false,
                    'error' => 'Interview not found',
                    'code' => 'NOT_FOUND',
                ], 404);
            }

            $validator = Validator::make($request->all(), [
                'status' => 'required|string',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'error' => 'Validation failed',
                    'code' => 'VALIDATION_ERROR',
                    'details' => $validator->errors()->toArray(),
                ], 400);
            }

            $interview->update([
                'status' => $request->input('status'),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Interview status updated successfully',
                'interview' => [
                    'id' => (string) $interview->id,
                    'status' => $interview->status,
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to update interview status',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Schedule an interview and send WhatsApp notification
     */
    public function schedule(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'candidates' => 'required|array|min:1',
                'candidates.*.id' => 'required|exists:form_responses,id',
                'candidates.*.name' => 'required|string|max:255',
                'candidates.*.phone' => 'required|string|max:50',
                'candidates.*.referenceId' => 'required|exists:form_responses,reference_id',
                'date' => 'required|date',
                'time' => 'required|date_format:H:i',
                'datetime' => 'required|date_format:Y-m-d H:i',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'error' => 'Validation failed',
                    'code' => 'VALIDATION_ERROR',
                    'details' => $validator->errors()->toArray(),
                ], 400);
            }

            $validated = $validator->validated();
            $candidates = $validated['candidates'];
            $date = $validated['date'];
            $time = $validated['time'];

            $scheduledInterviews = [];
            $failedCandidates = [];

            foreach ($candidates as $candidate) {
                try {
                    // Mark the form response as viewed
                    $formResponse = FormResponse::find($candidate['id']);
                    if ($formResponse) {
                        $formResponse->update([
                            'viewed' => true,
                            'viewed_at' => now(),
                            'viewed_by' => auth()->id(),
                        ]);
                    }

                    // Create interview record
                    $interview = Interviews::create([
                        'response_id' => $candidate['id'],
                        'reference_id' => $candidate['referenceId'],
                        'candidate_name' => $candidate['name'],
                        'phone' => $candidate['phone'],
                        'date' => $date,
                        'time' => $time,
                        'status' => 'pending',
                        'candidate_status' => null,
                        'user_id' => auth()->id(),
                    ]);

                    // Send WhatsApp message using Meta Business API
                    $whatsappResult = $this->sendWhatsAppTemplate(
                        $interview->phone,
                        $interview->candidate_name,
                        $interview->reference_id,
                        \Carbon\Carbon::parse($date)->format('d/m/Y'),
                        \Carbon\Carbon::parse($time)->format('h:i A'),
                        $interview->id
                    );

                    // Update status based on WhatsApp send result
                    if ($whatsappResult['success']) {
                        $interview->update([
                            'status' => 'pending',
                            'whatsapp_message_id' => $whatsappResult['message_id'],
                            'whatsapp_status' => 'sent',
                            'whatsapp_sent_at' => now(),
                        ]);
                        $scheduledInterviews[] = [
                            'id' => (string) $interview->id,
                            'referenceId' => $interview->reference_id,
                            'candidate_name' => $interview->candidate_name,
                            'name' => $interview->candidate_name,
                            'phone' => $interview->phone,
                            'date' => $interview->date->format('Y-m-d'),
                            'time' => $interview->time->format('H:i'),
                            'status' => $interview->status,
                            'candidate_status' => $interview->candidate_status,
                            'whatsapp_status' => $interview->whatsapp_status,
                            'whatsapp_message_id' => $interview->whatsapp_message_id,
                        ];
                    } else {
                        $interview->update([
                            'status' => 'failed',
                            'whatsapp_status' => 'failed',
                            'whatsapp_failed_at' => now(),
                        ]);
                        $failedCandidates[] = [
                            'id' => (string) $interview->id,
                            'referenceId' => $interview->reference_id,
                            'candidate_name' => $interview->candidate_name,
                            'name' => $interview->candidate_name,
                            'phone' => $interview->phone,
                            'date' => $interview->date->format('Y-m-d'),
                            'time' => $interview->time->format('H:i'),
                            'status' => $interview->status,
                            'candidate_status' => $interview->candidate_status,
                            'whatsapp_status' => $interview->whatsapp_status,
                            'error' => 'WhatsApp notification failed',
                        ];
                    }
                } catch (\Exception $e) {
                    Log::error('Failed to schedule interview for candidate: ' . $candidate['name'], [
                        'error' => $e->getMessage()
                    ]);
                    $failedCandidates[] = [
                        'referenceId' => $candidate['referenceId'],
                        'candidate_name' => $candidate['name'],
                        'error' => 'Failed to create interview record',
                    ];
                }
            }

            $totalScheduled = count($scheduledInterviews);
            $totalFailed = count($failedCandidates);

            return response()->json([
                'success' => true,
                'data' => [
                    'message' => "{$totalScheduled} interview(s) scheduled successfully, {$totalFailed} failed",
                    'scheduled' => $scheduledInterviews,
                    'failed' => $failedCandidates,
                    'summary' => [
                        'total' => count($candidates),
                        'scheduled' => $totalScheduled,
                        'failed' => $totalFailed,
                    ],
                ],
            ], 201);
        } catch (\Exception $e) {
            Log::error('Failed to schedule interviews: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to schedule interviews',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Format phone number to international format without +
     * Examples:
     * +971593830053 -> 971593830053
     * 971593830053 -> 971593830053
     * 503830053 -> 971503830053 (assumes UAE)
     * +1234567890 -> 1234567890
     */
    private function formatPhoneNumber(string $phone): string
    {
        // Remove all non-numeric characters (including +, spaces, dashes, etc.)
        $cleanPhone = preg_replace('/[^0-9]/', '', $phone);
        
        // If the number doesn't start with a country code (less than 10 digits or starts with 5,0,etc for UAE local numbers)
        // Assume it's a UAE number and add 971
        if (strlen($cleanPhone) < 10 || (strlen($cleanPhone) <= 10 && preg_match('/^[5|0]/', $cleanPhone))) {
            // Remove leading 0 if present (common in local UAE numbers)
            $cleanPhone = ltrim($cleanPhone, '0');
            // Add UAE country code
            $cleanPhone = '971' . $cleanPhone;
        }
        
        return $cleanPhone;
    }

    /**
     * Send WhatsApp template message using Meta Business API
     */
    private function sendWhatsAppTemplate(string $phone, string $firstName, string $referenceId, string $date, string $time, ?int $interviewId = null): array
    {
        try {
            $apiUrl = config('services.whatsapp.api_url');
            $apiKey = config('services.whatsapp.api_key');
            $phoneNumberId = config('services.whatsapp.phone_number_id');
            $templateName = config('services.whatsapp.interview_template_name', 'interview_confirmation');

            if (!$apiUrl || !$apiKey || !$phoneNumberId) {
                Log::warning('WhatsApp API credentials not configured');
                return ['success' => false, 'message_id' => null];
            }

            // Format phone number to ensure correct format
            $cleanPhone = $this->formatPhoneNumber($phone);

            $templateData = [
                'messaging_product' => 'whatsapp',
                'to' => $cleanPhone,
                'type' => 'template',
                'template' => [
                    'name' => $templateName,
                    'language' => [
                        'code' => 'en'
                    ],
                    'components' => [
                        [
                            'type' => 'body',
                            'parameters' => [
                                [
                                    'type' => 'text',
                                    'parameter_name' => 'first_name',
                                    'text' => $firstName
                                ],
                                [
                                    'type' => 'text',
                                    'parameter_name' => 'id',
                                    'text' => $referenceId
                                ],
                                [
                                    'type' => 'text',
                                    'parameter_name' => 'date',
                                    'text' => $date
                                ],
                                [
                                    'type' => 'text',
                                    'parameter_name' => 'time',
                                    'text' => $time
                                ]
                            ]
                        ]
                    ]
                ]
            ];

            // Prepare WhatsApp API request
            $response = Http::withToken($apiKey)
                ->post("{$apiUrl}/{$phoneNumberId}/messages", $templateData);

            if ($response->successful()) {
                $responseData = $response->json();
                $messageId = $responseData['messages'][0]['id'] ?? null;
                
                // Store message in database
                WhatsAppMessage::create([
                    'phone' => $cleanPhone,
                    'message_id' => $messageId,
                    'direction' => 'outbound',
                    'type' => 'template',
                    'message_body' => "Interview scheduled for {$firstName} on {$date} at {$time}",
                    'message_data' => $templateData,
                    'status' => 'sent',
                    'interview_id' => $interviewId,
                    'sent_at' => now(),
                ]);
                
                Log::info('WhatsApp template sent successfully', [
                    'phone' => $cleanPhone,
                    'message_id' => $messageId,
                    'response' => $responseData
                ]);
                
                return ['success' => true, 'message_id' => $messageId];
            } else {
                Log::error('WhatsApp API error', [
                    'phone' => $cleanPhone,
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);
                return ['success' => false, 'message_id' => null];
            }
        } catch (\Exception $e) {
            Log::error('WhatsApp send exception: ' . $e->getMessage());
            return ['success' => false, 'message_id' => null];
        }
    }

    /**
     * Send WhatsApp text message
     */
    private function sendWhatsAppMessage(string $phone, string $message, ?int $interviewId = null): bool
    {
        try {
            $apiUrl = config('services.whatsapp.api_url');
            $apiKey = config('services.whatsapp.api_key');
            $phoneNumberId = config('services.whatsapp.phone_number_id');

            if (!$apiUrl || !$apiKey || !$phoneNumberId) {
                Log::warning('WhatsApp API credentials not configured');
                return false;
            }

            // Format phone number to ensure correct format
            $cleanPhone = $this->formatPhoneNumber($phone);

            $messageData = [
                'messaging_product' => 'whatsapp',
                'to' => $cleanPhone,
                'type' => 'text',
                'text' => [
                    'body' => $message
                ]
            ];

            // Prepare WhatsApp API request
            $response = Http::withToken($apiKey)
                ->post("{$apiUrl}/{$phoneNumberId}/messages", $messageData);

            if ($response->successful()) {
                $responseData = $response->json();
                $messageId = $responseData['messages'][0]['id'] ?? null;

                // Store message in database
                WhatsAppMessage::create([
                    'phone' => $cleanPhone,
                    'message_id' => $messageId,
                    'direction' => 'outbound',
                    'type' => 'text',
                    'message_body' => $message,
                    'message_data' => $messageData,
                    'status' => 'sent',
                    'interview_id' => $interviewId,
                    'sent_at' => now(),
                ]);

                Log::info('WhatsApp message sent successfully', [
                    'phone' => $cleanPhone,
                    'message_id' => $messageId,
                    'message' => $message
                ]);
                return true;
            } else {
                Log::error('WhatsApp message API error', [
                    'phone' => $cleanPhone,
                    'status' => $response->status(),
                    'response' => $response->body()
                ]);
                return false;
            }
        } catch (\Exception $e) {
            Log::error('WhatsApp message send exception: ' . $e->getMessage());
            return false;
        }
    }

    /**
     * Handle WhatsApp webhook verification (GET request)
     */
    public function verifyWebhook(Request $request): JsonResponse
    {
        try {
            $verifyToken = config('services.whatsapp.webhook_verify_token');
            $mode = $request->query('hub_mode');
            $token = $request->query('hub_verify_token');
            $challenge = $request->query('hub_challenge');

            if ($mode === 'subscribe' && $token === $verifyToken) {
                Log::info('WhatsApp webhook verified');
                return response()->json((int)$challenge, 200);
            }

            Log::warning('WhatsApp webhook verification failed', [
                'mode' => $mode,
                'token' => $token
            ]);

            return response()->json([
                'success' => false,
                'error' => 'Verification failed'
            ], 403);
        } catch (\Exception $e) {
            Log::error('Webhook verification exception: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Verification failed'
            ], 500);
        }
    }

    /**
     * Handle WhatsApp webhook for incoming messages (POST request)
     */
    public function handleWebhook(Request $request): JsonResponse
    {
        try {
            Log::info('WhatsApp webhook received', ['payload' => $request->all()]);

            $data = $request->all();

            // Check if this is a status update webhook
            if (isset($data['entry'][0]['changes'][0]['value']['statuses'])) {
                return $this->handleStatusUpdates($data);
            }

            // Validate webhook structure for messages
            if (!isset($data['entry'][0]['changes'][0]['value']['messages'][0])) {
                Log::info('WhatsApp webhook - no messages found');
                return response()->json(['success' => true], 200);
            }

            $value = $data['entry'][0]['changes'][0]['value'];
            $message = $value['messages'][0];
            $from = $message['from'];
            $messageType = $message['type'];

            Log::info('WhatsApp message received', [
                'from' => $from,
                'type' => $messageType,
                'message' => $message
            ]);

            // Handle button responses (new format from Meta)
            if ($messageType === 'button') {
                $buttonPayload = strtolower($message['button']['payload']);
                $buttonText = $message['button']['text'];

                Log::info('WhatsApp button received', [
                    'from' => $from,
                    'payload' => $buttonPayload,
                    'text' => $buttonText
                ]);

                // Find interview by phone number
                $cleanPhone = $this->formatPhoneNumber($from);
                $interview = Interviews::where('phone', 'LIKE', '%' . substr($cleanPhone, -9))
                    ->orderBy('created_at', 'desc')
                    ->first();

                if (!$interview) {
                    Log::warning('WhatsApp webhook - interview not found for phone', ['phone' => $from]);
                    $this->sendWhatsAppMessage($from, "We couldn't find your interview record. Please contact us directly.");
                    return response()->json(['success' => true], 200);
                }

                // Process based on button payload
                $responseMessage = '';
                switch ($buttonPayload) {
                    case 'confirm':
                        $interview->update(['status' => 'confirmed']);
                        $responseMessage = "✅ Thank you for confirming your interview!\n\n";
                        $responseMessage .= "📅 Date: {$interview->date->format('d/m/Y')}\n";
                        $responseMessage .= "🕐 Time: {$interview->time->format('h:i A')}\n\n";
                        $responseMessage .= "We look forward to meeting you!\n";
                        $responseMessage .= "Reference: {$interview->reference_id}";
                        Log::info('Interview confirmed', ['interview_id' => $interview->id, 'phone' => $from]);
                        break;

                    case 'reschedule':
                        $interview->update(['status' => 'pending']);
                        $responseMessage = "📝 We understand you need to reschedule your interview.\n\n";
                        $responseMessage .= "Our team will contact you shortly to arrange a new time.\n\n";
                        $responseMessage .= "Reference: {$interview->reference_id}";
                        Log::info('Interview reschedule requested', ['interview_id' => $interview->id, 'phone' => $from]);
                        break;

                    case 'decline':
                        $interview->update(['status' => 'cancelled']);
                        $responseMessage = "❌ Interview Cancelled\n\n";
                        $responseMessage .= "Thank you for letting us know. We're sorry you can't make it.\n\n";
                        $responseMessage .= "If you change your mind or would like to apply in the future, please feel free to contact us.\n\n";
                        $responseMessage .= "Reference: {$interview->reference_id}";
                        Log::info('Interview declined', ['interview_id' => $interview->id, 'phone' => $from]);
                        break;

                    default:
                        Log::warning('WhatsApp webhook - unknown button payload', ['payload' => $buttonPayload]);
                        $responseMessage = "Thank you for your response. Our team will get back to you soon.";
                }

                // Store inbound message
                WhatsAppMessage::create([
                    'phone' => $from,
                    'message_id' => $message['id'],
                    'direction' => 'inbound',
                    'type' => 'button',
                    'message_body' => $buttonText,
                    'message_data' => $message,
                    'status' => 'received',
                    'interview_id' => $interview->id,
                ]);

                // Send confirmation message back to candidate
                $this->sendWhatsAppMessage($from, $responseMessage, $interview->id);
            }
            // Handle interactive button responses (old format)
            elseif ($messageType === 'interactive') {
                $buttonReply = $message['interactive']['button_reply'];
                $buttonId = $buttonReply['id'];
                $buttonTitle = $buttonReply['title'];

                Log::info('WhatsApp interactive button received', [
                    'from' => $from,
                    'button_id' => $buttonId,
                    'button_title' => $buttonTitle
                ]);

                // Extract reference ID from button ID (format: action_referenceId)
                $parts = explode('_', $buttonId);
                $action = $parts[0] ?? null;
                $referenceId = $parts[1] ?? null;

                if (!$referenceId) {
                    Log::warning('WhatsApp webhook - no reference ID found in button', ['button_id' => $buttonId]);
                    return response()->json(['success' => true], 200);
                }

                // Find interview by reference ID
                $interview = Interviews::where('reference_id', $referenceId)->first();

                if (!$interview) {
                    Log::warning('WhatsApp webhook - interview not found', ['reference_id' => $referenceId]);
                    return response()->json(['success' => true], 200);
                }

                // Process based on action
                $responseMessage = '';
                switch ($action) {
                    case 'confirm':
                        $interview->update(['status' => 'confirmed']);
                        $responseMessage = "✅ Thank you for confirming your interview!\n\n";
                        $responseMessage .= "📅 Date: {$interview->date->format('d/m/Y')}\n";
                        $responseMessage .= "🕐 Time: {$interview->time->format('h:i A')}\n\n";
                        $responseMessage .= "We look forward to meeting you!\n";
                        $responseMessage .= "Reference: {$interview->reference_id}";
                        Log::info('Interview confirmed', ['interview_id' => $interview->id, 'reference_id' => $referenceId]);
                        break;

                    case 'reschedule':
                        $interview->update(['status' => 'pending']);
                        $responseMessage = "📝 We understand you need to reschedule your interview.\n\n";
                        $responseMessage .= "Our team will contact you shortly to arrange a new time.\n\n";
                        $responseMessage .= "Reference: {$interview->reference_id}";
                        Log::info('Interview reschedule requested', ['interview_id' => $interview->id, 'reference_id' => $referenceId]);
                        break;

                    case 'decline':
                        $interview->update(['status' => 'cancelled']);
                        $responseMessage = "❌ Interview Cancelled\n\n";
                        $responseMessage .= "Thank you for letting us know. We're sorry you can't make it.\n\n";
                        $responseMessage .= "If you change your mind or would like to apply in the future, please feel free to contact us.\n\n";
                        $responseMessage .= "Reference: {$interview->reference_id}";
                        Log::info('Interview declined', ['interview_id' => $interview->id, 'reference_id' => $referenceId]);
                        break;

                    default:
                        Log::warning('WhatsApp webhook - unknown action', ['action' => $action]);
                        return response()->json(['success' => true], 200);
                }

                // Store inbound message
                WhatsAppMessage::create([
                    'phone' => $from,
                    'message_id' => $message['id'],
                    'direction' => 'inbound',
                    'type' => 'interactive',
                    'message_body' => $buttonTitle,
                    'message_data' => $message,
                    'status' => 'received',
                    'interview_id' => $interview->id,
                ]);

                // Send confirmation message back to candidate
                $this->sendWhatsAppMessage($from, $responseMessage, $interview->id);
            }
            // Handle text message responses
            elseif ($messageType === 'text') {
                $messageBody = strtolower($message['text']['body']);
                
                Log::info('WhatsApp text message received', [
                    'from' => $from,
                    'message' => $messageBody
                ]);

                // Try to find interview by phone number
                $cleanPhone = $this->formatPhoneNumber($from);
                $interview = Interviews::where('phone', 'LIKE', '%' . substr($cleanPhone, -9))
                    ->orderBy('created_at', 'desc')
                    ->first();

                if ($interview) {
                    // Store inbound message
                    WhatsAppMessage::create([
                        'phone' => $from,
                        'message_id' => $message['id'],
                        'direction' => 'inbound',
                        'type' => 'text',
                        'message_body' => $message['text']['body'],
                        'message_data' => $message,
                        'status' => 'received',
                        'interview_id' => $interview->id,
                    ]);

                    $responseMessage = '';
                    
                    if (str_contains($messageBody, 'confirm') || str_contains($messageBody, 'yes')) {
                        $interview->update(['status' => 'confirmed']);
                        $responseMessage = "✅ Thank you for confirming your interview!\n\n";
                        $responseMessage .= "📅 Date: {$interview->date->format('d/m/Y')}\n";
                        $responseMessage .= "🕐 Time: {$interview->time->format('h:i A')}\n\n";
                        $responseMessage .= "We look forward to meeting you!\n";
                        $responseMessage .= "Reference: {$interview->reference_id}";
                    } elseif (str_contains($messageBody, 'reschedule') || str_contains($messageBody, 'change')) {
                        $interview->update(['status' => 'pending']);
                        $responseMessage = "📝 We understand you need to reschedule your interview.\n\n";
                        $responseMessage .= "Our team will contact you shortly to arrange a new time.\n\n";
                        $responseMessage .= "Reference: {$interview->reference_id}";
                    } elseif (str_contains($messageBody, 'decline') || str_contains($messageBody, 'cancel') || str_contains($messageBody, 'no')) {
                        $interview->update(['status' => 'cancelled']);
                        $responseMessage = "❌ Interview Cancelled\n\n";
                        $responseMessage .= "Thank you for letting us know. We're sorry you can't make it.\n\n";
                        $responseMessage .= "If you change your mind or would like to apply in the future, please feel free to contact us.\n\n";
                        $responseMessage .= "Reference: {$interview->reference_id}";
                    }

                    if ($responseMessage) {
                        $this->sendWhatsAppMessage($from, $responseMessage, $interview->id);
                    }
                }
            }

            return response()->json(['success' => true], 200);
        } catch (\Exception $e) {
            Log::error('WhatsApp webhook exception: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json(['success' => true], 200); // Always return 200 to avoid Meta retrying
        }
    }

    /**
     * Handle WhatsApp message status updates
     */
    private function handleStatusUpdates(array $data): JsonResponse
    {
        try {
            // Validate webhook structure for status updates
            if (!isset($data['entry'][0]['changes'][0]['value']['statuses'][0])) {
                Log::info('WhatsApp status webhook - no statuses found');
                return response()->json(['success' => true], 200);
            }

            $statuses = $data['entry'][0]['changes'][0]['value']['statuses'];

            foreach ($statuses as $status) {
                $messageId = $status['id'];
                $statusType = $status['status']; // sent, delivered, read, failed
                $timestamp = $status['timestamp'] ?? null;

                Log::info('WhatsApp status update', [
                    'message_id' => $messageId,
                    'status' => $statusType,
                    'timestamp' => $timestamp
                ]);

                // Find interview by WhatsApp message ID
                $interview = Interviews::where('whatsapp_message_id', $messageId)->first();

                if (!$interview) {
                    Log::warning('WhatsApp status webhook - interview not found for message', ['message_id' => $messageId]);
                    continue;
                }

                // Update interview based on status
                $updateData = ['whatsapp_status' => $statusType];

                switch ($statusType) {
                    case 'sent':
                        $updateData['whatsapp_sent_at'] = $timestamp ? \Carbon\Carbon::createFromTimestamp($timestamp) : now();
                        break;

                    case 'delivered':
                        $updateData['whatsapp_delivered_at'] = $timestamp ? \Carbon\Carbon::createFromTimestamp($timestamp) : now();
                        break;

                    case 'read':
                        $updateData['whatsapp_read_at'] = $timestamp ? \Carbon\Carbon::createFromTimestamp($timestamp) : now();
                        break;

                    case 'failed':
                        $updateData['whatsapp_failed_at'] = $timestamp ? \Carbon\Carbon::createFromTimestamp($timestamp) : now();
                        $errorCode = $status['errors'][0]['code'] ?? null;
                        $errorTitle = $status['errors'][0]['title'] ?? null;
                        Log::error('WhatsApp message failed', [
                            'interview_id' => $interview->id,
                            'message_id' => $messageId,
                            'error_code' => $errorCode,
                            'error_title' => $errorTitle
                        ]);
                        break;
                }

                $interview->update($updateData);

                // Also update the message in whatsapp_messages table
                $whatsappMessage = WhatsAppMessage::where('message_id', $messageId)->first();
                if ($whatsappMessage) {
                    $messageUpdateData = ['status' => $statusType];
                    
                    switch ($statusType) {
                        case 'sent':
                            $messageUpdateData['sent_at'] = $timestamp ? \Carbon\Carbon::createFromTimestamp($timestamp) : now();
                            break;
                        case 'delivered':
                            $messageUpdateData['delivered_at'] = $timestamp ? \Carbon\Carbon::createFromTimestamp($timestamp) : now();
                            break;
                        case 'read':
                            $messageUpdateData['read_at'] = $timestamp ? \Carbon\Carbon::createFromTimestamp($timestamp) : now();
                            break;
                        case 'failed':
                            $messageUpdateData['failed_at'] = $timestamp ? \Carbon\Carbon::createFromTimestamp($timestamp) : now();
                            break;
                    }
                    
                    $whatsappMessage->update($messageUpdateData);
                }

                Log::info('Interview WhatsApp status updated', [
                    'interview_id' => $interview->id,
                    'status' => $statusType
                ]);
            }

            return response()->json(['success' => true], 200);
        } catch (\Exception $e) {
            Log::error('WhatsApp status webhook exception: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json(['success' => true], 200); // Always return 200 to avoid Meta retrying
        }
    }

    /**
     * Get WhatsApp messages for a phone number
     */
    public function getMessages(Request $request): JsonResponse
    {
        try {
            $phone = $request->input('phone');
            $interviewId = $request->input('interview_id');
            $limit = $request->input('limit', 50);

            $query = WhatsAppMessage::query();

            if ($phone) {
                $cleanPhone = $this->formatPhoneNumber($phone);
                $query->where('phone', 'LIKE', '%' . substr($cleanPhone, -9) . '%');
            }

            if ($interviewId) {
                $query->where('interview_id', $interviewId);
            }

            $messages = $query->orderBy('created_at', 'desc')
                ->limit($limit)
                ->get();

            $formattedMessages = $messages->map(function ($message) {
                return [
                    'id' => $message->id,
                    'phone' => $message->phone,
                    'message_id' => $message->message_id,
                    'direction' => $message->direction,
                    'type' => $message->type,
                    'message_body' => $message->message_body,
                    'status' => $message->status,
                    'interview_id' => $message->interview_id,
                    'sent_at' => $message->sent_at?->format('Y-m-d H:i:s'),
                    'delivered_at' => $message->delivered_at?->format('Y-m-d H:i:s'),
                    'read_at' => $message->read_at?->format('Y-m-d H:i:s'),
                    'failed_at' => $message->failed_at?->format('Y-m-d H:i:s'),
                    'created_at' => $message->created_at->format('Y-m-d H:i:s'),
                ];
            });

            return response()->json([
                'success' => true,
                'data' => [
                    'messages' => $formattedMessages,
                    'total' => $messages->count(),
                ],
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to fetch messages: ' . $e->getMessage());
            return response()->json([
                'success' => false,
                'error' => 'Failed to fetch messages',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Get customer profile by phone number
     * Returns all responses, interviews, and WhatsApp messages
     */
    public function getCustomerProfile(Request $request): JsonResponse
    {
        try {
            $phone = $request->input('phone');

            if (!$phone) {
                return response()->json([
                    'success' => false,
                    'error' => 'Phone number is required',
                    'code' => 'VALIDATION_ERROR',
                ], 400);
            }

            // Format phone number
            $cleanPhone = $this->formatPhoneNumber($phone);
            $phonePattern = '%' . substr($cleanPhone, -9) . '%';

            // Get all form responses for this phone number
            $responses = FormResponse::whereRaw("JSON_UNQUOTE(JSON_EXTRACT(data, '$.phone')) LIKE ?", [$phonePattern])
                ->orderBy('created_at', 'desc')
                ->get();

            $formattedResponses = $responses->map(function ($response) {
                return [
                    'id' => (string) $response->id,
                    'referenceId' => $response->reference_id,
                    'timestamp' => $response->created_at->format('Y-m-d H:i:s'),
                    'viewed' => $response->viewed,
                    'viewedAt' => $response->viewed_at?->format('Y-m-d H:i:s'),
                    'data' => $response->data,
                ];
            });

            // Get all interviews for this phone number
            $interviews = Interviews::where('phone', 'LIKE', $phonePattern)
                ->orderBy('created_at', 'desc')
                ->get();

            $formattedInterviews = $interviews->map(function ($interview) {
                return [
                    'id' => (string) $interview->id,
                    'response_id' => (string) $interview->response_id,
                    'referenceId' => $interview->reference_id,
                    'candidate_name' => $interview->candidate_name,
                    'phone' => $interview->phone,
                    'date' => $interview->date->format('Y-m-d'),
                    'time' => $interview->time->format('H:i'),
                    'status' => $interview->status,
                    'candidate_status' => $interview->candidate_status,
                    'whatsapp_status' => $interview->whatsapp_status,
                    'whatsapp_sent_at' => $interview->whatsapp_sent_at?->format('Y-m-d H:i:s'),
                    'whatsapp_delivered_at' => $interview->whatsapp_delivered_at?->format('Y-m-d H:i:s'),
                    'whatsapp_read_at' => $interview->whatsapp_read_at?->format('Y-m-d H:i:s'),
                    'created_at' => $interview->created_at->format('Y-m-d H:i:s'),
                ];
            });

            // Get all WhatsApp messages for this phone number
            $messages = WhatsAppMessage::where('phone', 'LIKE', $phonePattern)
                ->orderBy('created_at', 'desc')
                ->get();

            $formattedMessages = $messages->map(function ($message) {
                return [
                    'id' => $message->id,
                    'message_id' => $message->message_id,
                    'direction' => $message->direction,
                    'type' => $message->type,
                    'message_body' => $message->message_body,
                    'status' => $message->status,
                    'interview_id' => $message->interview_id,
                    'sent_at' => $message->sent_at?->format('Y-m-d H:i:s'),
                    'delivered_at' => $message->delivered_at?->format('Y-m-d H:i:s'),
                    'read_at' => $message->read_at?->format('Y-m-d H:i:s'),
                    'failed_at' => $message->failed_at?->format('Y-m-d H:i:s'),
                    'created_at' => $message->created_at->format('Y-m-d H:i:s'),
                ];
            });

            // Extract customer information from the most recent response
            $customerInfo = null;
            if ($responses->isNotEmpty()) {
                $latestResponse = $responses->first();
                $data = $latestResponse->data;
                $customerInfo = [
                    'name' => $data['name'] ?? null,
                    'phone' => $data['phone'] ?? $cleanPhone,
                    'email' => $data['email'] ?? null,
                    'nationality' => $data['nationality'] ?? null,
                    'gender' => $data['gender'] ?? null,
                    'position' => $data['position'] ?? null,
                    'experience' => $data['experience'] ?? null,
                    'english_level' => $data['english_level'] ?? null,
                    'in_dubai' => $data['in_dubai'] ?? null,
                ];
            }

            // Calculate statistics
            $stats = [
                'total_responses' => $responses->count(),
                'total_interviews' => $interviews->count(),
                'total_messages' => $messages->count(),
                'interviews_by_status' => [
                    'pending' => $interviews->where('status', 'pending')->count(),
                    'confirmed' => $interviews->where('status', 'confirmed')->count(),
                    'cancelled' => $interviews->where('status', 'cancelled')->count(),
                    'completed' => $interviews->where('status', 'completed')->count(),
                    'failed' => $interviews->where('status', 'failed')->count(),
                ],
                'candidate_status' => $interviews->where('candidate_status', '!=', null)->first()?->candidate_status,
                'last_interaction' => max(
                    $responses->max('created_at'),
                    $interviews->max('created_at'),
                    $messages->max('created_at')
                )?->format('Y-m-d H:i:s'),
            ];

            return response()->json([
                'success' => true,
                'data' => [
                    'phone' => $cleanPhone,
                    'customer_info' => $customerInfo,
                    'stats' => $stats,
                    'responses' => $formattedResponses,
                    'interviews' => $formattedInterviews,
                    'messages' => $formattedMessages,
                ],
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to fetch customer profile: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'error' => 'Failed to fetch customer profile',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Get all conversations list
     * Groups messages by phone number and shows latest message with unread count
     */
    public function getConversations(Request $request): JsonResponse
    {
        try {
            $limit = $request->input('limit', 50);
            $search = $request->input('search'); // Search by phone or name

            // Get all unique phone numbers with their latest message
            $conversations = WhatsAppMessage::selectRaw('
                phone,
                MAX(created_at) as last_message_time,
                (SELECT message_body FROM whatsapp_messages wm2 
                 WHERE wm2.phone = whatsapp_messages.phone 
                 ORDER BY created_at DESC LIMIT 1) as last_message,
                (SELECT direction FROM whatsapp_messages wm3 
                 WHERE wm3.phone = whatsapp_messages.phone 
                 ORDER BY created_at DESC LIMIT 1) as last_message_direction,
                (SELECT created_at FROM whatsapp_messages wm4 
                 WHERE wm4.phone = whatsapp_messages.phone 
                 AND wm4.direction = "inbound"
                 ORDER BY created_at DESC LIMIT 1) as last_inbound_time
            ')
            ->groupBy('phone')
            ->orderByDesc('last_message_time')
            ->limit($limit)
            ->get();

            $formattedConversations = $conversations->map(function ($conversation) use ($search) {
                // Get customer name from latest form response
                $cleanPhone = $this->formatPhoneNumber($conversation->phone);
                $phonePattern = '%' . substr($cleanPhone, -9) . '%';
                
                $response = FormResponse::whereRaw("JSON_UNQUOTE(JSON_EXTRACT(data, '$.phone')) LIKE ?", [$phonePattern])
                    ->orderBy('created_at', 'desc')
                    ->first();
                
                $name = $response && isset($response->data['name']) ? $response->data['name'] : null;

                // If search is provided, filter by phone or name
                if ($search) {
                    $searchLower = strtolower($search);
                    $phoneLower = strtolower($conversation->phone);
                    $nameLower = strtolower($name ?? '');
                    
                    if (strpos($phoneLower, $searchLower) === false && strpos($nameLower, $searchLower) === false) {
                        return null; // Skip this conversation
                    }
                }

                // Count unread messages (inbound messages after last outbound message)
                $lastOutboundMessage = WhatsAppMessage::where('phone', $conversation->phone)
                    ->where('direction', 'outbound')
                    ->orderBy('created_at', 'desc')
                    ->first();

                $unreadCount = 0;
                if ($lastOutboundMessage) {
                    $unreadCount = WhatsAppMessage::where('phone', $conversation->phone)
                        ->where('direction', 'inbound')
                        ->where('created_at', '>', $lastOutboundMessage->created_at)
                        ->count();
                } else {
                    // No outbound message yet, count all inbound messages
                    $unreadCount = WhatsAppMessage::where('phone', $conversation->phone)
                        ->where('direction', 'inbound')
                        ->count();
                }

                // Calculate hours since last received message
                $hoursSinceLastReceived = null;
                $canReply = false;
                
                if ($conversation->last_inbound_time) {
                    $lastInboundTime = \Carbon\Carbon::parse($conversation->last_inbound_time);
                    $hoursSinceLastReceived = round($lastInboundTime->diffInMinutes(now()) / 60, 1);
                    
                    // Can reply within 24 hours of last inbound message
                    $canReply = $hoursSinceLastReceived < 24;
                }

                return [
                    'phone' => $conversation->phone,
                    'name' => $name,
                    'last_message' => $conversation->last_message,
                    'last_message_time' => \Carbon\Carbon::parse($conversation->last_message_time)->format('Y-m-d H:i:s'),
                    'last_message_direction' => $conversation->last_message_direction,
                    'unread_count' => $unreadCount,
                    'can_reply' => $canReply,
                    'hours_since_last_received' => $hoursSinceLastReceived,
                ];
            })->filter()->values(); // Remove null values and re-index

            return response()->json([
                'success' => true,
                'data' => $formattedConversations,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to fetch conversations: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'error' => 'Failed to fetch conversations',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Get messages for a specific conversation
     * Returns all messages for a phone number, ordered oldest first
     */
    public function getChatMessages(Request $request): JsonResponse
    {
        try {
            $phone = $request->input('phone');
            $limit = $request->input('limit', 100);

            if (!$phone) {
                return response()->json([
                    'success' => false,
                    'error' => 'Phone number is required',
                    'code' => 'VALIDATION_ERROR',
                ], 400);
            }

            // Format phone number for search
            $cleanPhone = $this->formatPhoneNumber($phone);
            $phonePattern = '%' . substr($cleanPhone, -9) . '%';

            // Get all messages for this phone number, ordered oldest first
            $messages = WhatsAppMessage::where('phone', 'LIKE', $phonePattern)
                ->orderBy('created_at', 'asc')
                ->limit($limit)
                ->get();

            $formattedMessages = $messages->map(function ($message) {
                return [
                    'id' => $message->id,
                    'message_id' => $message->message_id,
                    'direction' => $message->direction,
                    'type' => $message->type,
                    'message_body' => $message->message_body,
                    'status' => $message->status,
                    'sent_at' => $message->sent_at?->format('Y-m-d H:i:s'),
                    'delivered_at' => $message->delivered_at?->format('Y-m-d H:i:s'),
                    'read_at' => $message->read_at?->format('Y-m-d H:i:s'),
                    'created_at' => $message->created_at->format('Y-m-d H:i:s'),
                ];
            });

            return response()->json([
                'success' => true,
                'data' => $formattedMessages,
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to fetch chat messages: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'error' => 'Failed to fetch chat messages',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }

    /**
     * Send a text message to a candidate
     * Validates 24-hour reply window before sending
     */
    public function sendChatMessage(Request $request): JsonResponse
    {
        try {
            $validator = Validator::make($request->all(), [
                'phone' => 'required|string',
                'message' => 'required|string|max:4096',
            ]);

            if ($validator->fails()) {
                return response()->json([
                    'success' => false,
                    'message' => 'Validation failed',
                    'error' => 'VALIDATION_ERROR',
                    'errors' => $validator->errors(),
                ], 400);
            }

            $phone = $request->input('phone');
            $message = $request->input('message');

            // Format phone number
            $cleanPhone = $this->formatPhoneNumber($phone);

            // Check 24-hour reply window
            $lastInboundMessage = WhatsAppMessage::where('phone', 'LIKE', '%' . substr($cleanPhone, -9) . '%')
                ->where('direction', 'inbound')
                ->orderBy('created_at', 'desc')
                ->first();

            if (!$lastInboundMessage) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot send message: No previous message from candidate',
                    'error' => 'NO_INBOUND_MESSAGE',
                ], 400);
            }

            $hoursSinceLastReceived = $lastInboundMessage->created_at->diffInHours(now());

            if ($hoursSinceLastReceived >= 24) {
                return response()->json([
                    'success' => false,
                    'message' => 'Cannot send message: 24-hour reply window has expired',
                    'error' => 'WINDOW_EXPIRED',
                    'hours_since_last_received' => round($hoursSinceLastReceived, 1),
                ], 400);
            }

            // Send message via WhatsApp API
            $whatsappConfig = config('services.whatsapp');
            $url = $whatsappConfig['api_url'] . '/' . $whatsappConfig['phone_number_id'] . '/messages';

            $response = Http::withHeaders([
                'Authorization' => 'Bearer ' . $whatsappConfig['api_key'],
                'Content-Type' => 'application/json',
            ])->post($url, [
                'messaging_product' => 'whatsapp',
                'recipient_type' => 'individual',
                'to' => $cleanPhone,
                'type' => 'text',
                'text' => [
                    'preview_url' => false,
                    'body' => $message,
                ],
            ]);

            if (!$response->successful()) {
                Log::error('WhatsApp API error when sending chat message', [
                    'status' => $response->status(),
                    'body' => $response->body(),
                    'phone' => $cleanPhone,
                ]);

                return response()->json([
                    'success' => false,
                    'message' => 'Failed to send message',
                    'error' => 'WHATSAPP_API_ERROR',
                ], 500);
            }

            $responseData = $response->json();
            $messageId = $responseData['messages'][0]['id'] ?? null;

            // Store message in database
            $whatsappMessage = WhatsAppMessage::create([
                'phone' => $cleanPhone,
                'message_id' => $messageId,
                'direction' => 'outbound',
                'type' => 'text',
                'message_body' => $message,
                'message_data' => $responseData,
                'status' => 'sent',
                'interview_id' => null,
                'sent_at' => now(),
            ]);

            return response()->json([
                'success' => true,
                'message' => 'Message sent successfully',
                'data' => [
                    'id' => $whatsappMessage->id,
                    'message_id' => $whatsappMessage->message_id,
                    'direction' => $whatsappMessage->direction,
                    'type' => $whatsappMessage->type,
                    'message_body' => $whatsappMessage->message_body,
                    'status' => $whatsappMessage->status,
                    'sent_at' => $whatsappMessage->sent_at->format('Y-m-d H:i:s'),
                    'delivered_at' => null,
                    'read_at' => null,
                    'created_at' => $whatsappMessage->created_at->format('Y-m-d H:i:s'),
                ],
            ]);
        } catch (\Exception $e) {
            Log::error('Failed to send chat message: ' . $e->getMessage(), [
                'trace' => $e->getTraceAsString()
            ]);
            return response()->json([
                'success' => false,
                'message' => 'Failed to send message',
                'error' => 'SERVER_ERROR',
            ], 500);
        }
    }
}
