<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\UpdateFormConfigRequest;
use App\Models\FormConfig;
use Illuminate\Http\JsonResponse;

class FormConfigController extends Controller
{
    /**
     * Normalize field options - use label as value if value is option_xxx or empty
     */
    private function normalizeFieldOptions(array $fields): array
    {
        foreach ($fields as &$field) {
            if (isset($field['options']) && is_array($field['options'])) {
                foreach ($field['options'] as &$option) {
                    // If value is option_xxx pattern or empty, use the label as value
                    if (empty($option['value']) || preg_match('/^option_\w+$/', $option['value'])) {
                        $option['value'] = $option['label'];
                    }
                }
            }
        }
        
        return $fields;
    }

    /**
     * Update form configuration
     */
    public function update(UpdateFormConfigRequest $request): JsonResponse
    {
        try {
            $validated = $request->validated();

            // Normalize option values - use label as value if value is option_xxx or empty
            $normalizedFields = $this->normalizeFieldOptions($validated['fields']);

            $config = FormConfig::latest()->first();
            
            $oldFields = $config ? $config->fields : [];

            if ($config) {
                $config->update([
                    'whatsapp_number' => $validated['whatsappNumber'],
                    'fields' => $normalizedFields,
                    'updated_by' => auth()->id(),
                ]);
            } else {
                $config = FormConfig::create([
                    'whatsapp_number' => $validated['whatsappNumber'],
                    'fields' => $normalizedFields,
                    'updated_by' => auth()->id(),
                ]);
            }

            // Calculate changes
            $fieldsModified = 0;
            $fieldsEnabled = 0;
            $fieldsDisabled = 0;

            foreach ($validated['fields'] as $field) {
                if ($field['enabled']) {
                    $fieldsEnabled++;
                } else {
                    $fieldsDisabled++;
                }
            }

            if (count($oldFields) > 0) {
                $fieldsModified = count($validated['fields']);
            }

            return response()->json([
                'success' => true,
                'data' => [
                    'message' => 'Configuration updated successfully',
                    'updatedAt' => $config->updated_at->toIso8601String(),
                    'updatedBy' => (string) auth()->id(),
                    'changes' => [
                        'whatsappNumber' => $config->wasChanged('whatsapp_number') ? 'updated' : 'unchanged',
                        'fieldsModified' => $fieldsModified,
                        'fieldsEnabled' => $fieldsEnabled,
                        'fieldsDisabled' => $fieldsDisabled,
                    ],
                ],
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Failed to update configuration',
                'code' => 'SERVER_ERROR',
            ], 500);
        }
    }
}
