<?php

namespace App\Http\Controllers\Api\Admin;

use App\Http\Controllers\Controller;
use App\Http\Requests\AdminLoginRequest;
use App\Models\User;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Hash;

class AuthController extends Controller
{
    /**
     * Admin authentication
     */
    public function login(AdminLoginRequest $request): JsonResponse
    {
        $validated = $request->validated();
        
        // Find user by email (username field)
        $user = User::where('email', $validated['username'])->first();

        // Check if account is locked
        if ($user && $user->isLocked()) {
            return response()->json([
                'success' => false,
                'error' => 'Account has been locked due to multiple failed attempts',
                'code' => 'ACCOUNT_LOCKED',
                'details' => [
                    'lockedUntil' => $user->locked_until->toIso8601String(),
                ],
            ], 403);
        }

        // Validate credentials
        if (!$user || !Hash::check($validated['password'], $user->password)) {
            if ($user) {
                $user->incrementFailedAttempts();
            }
            
            return response()->json([
                'success' => false,
                'error' => 'Invalid username or password',
                'code' => 'INVALID_CREDENTIALS',
            ], 401);
        }

        // Reset failed attempts on successful login
        $user->resetFailedAttempts();

        // Create token
        $token = $user->createToken('admin-token')->plainTextToken;

        return response()->json([
            'success' => true,
            'data' => [
                'token' => $token,
                'user' => [
                    'id' => (string) $user->id,
                    'username' => $user->email,
                    'role' => $user->role,
                ],
                'expiresIn' => 86400, // 24 hours in seconds
            ],
        ]);
    }

    /**
     * Logout
     */
    public function logout(): JsonResponse
    {
        auth()->user()->currentAccessToken()->delete();

        return response()->json([
            'success' => true,
            'message' => 'Logged out successfully',
        ]);
    }
}
